#include "tokenizer.h"
#include <memory>

namespace lp {

	// TODO: clean up eof(): read with while( cin >> ch ) style instead
	Dictionary prolog_dic = Dictionary(Dictionary::Prolog_syntax());

	Functor* stream_to_expr(
		istream& is, 
		long long& linec, 
		std::set<id_type>& qvars,
		const Dictionary& dic) 
	{
		//cerr << "In stream_to_expr\n";
		// Tokenize will increase line count, store backup
		const long long init_line = linec;
		list<Token> tokenq = tokenize(is, linec, qvars, dic);
		auto at = find_if(tokenq.begin(),tokenq.end(),[&](const Token& t){return !t.is_newline();});
		if (at == tokenq.end()) {
			return nullptr; // end of input (there's only newlines)
		}
		
		//if (at->id() == if_id && find_if(++at,tokenq.end(),[&](const Token& t){return !t.is_newline();}) == tokenq.end()) {
		//	// Special case of constant ":-" => create empty clause
		//	return new Functor(if_id);
		//}

		//cerr << "tokenq: ";
		//for_each(tokenq.cbegin(), tokenq.cend(), [](const Token t){cerr << t.symbol() << " ";});
		//cerr << "\n";

		long long linec2 = init_line;
		tokens_to_operators(tokenq, linec2, dic);

		//cerr << "tokenq2: ";
		//for_each(tokenq.cbegin(), tokenq.cend(), [](const Token t){cerr << t.symbol() << " ";});
		//cerr << "\n";

		linec2 = init_line;
		list<Token> rpn = infix_to_postfix(tokenq,linec2);

		//cerr << "tokenq3: ";
		//for_each(tokenq.cbegin(), tokenq.cend(), [](const Token t){cerr << t.symbol() << " ";});
		//cerr << "\n";

		// Strip off newline tokens
		for (auto i = tokenq.begin(); i != tokenq.end(); ) {
			if (i->is_newline()) i = tokenq.erase(i);
			else ++i;
		}

		//cerr << "tokenq4: ";
		//for_each(tokenq.cbegin(), tokenq.cend(), [](const Token t){cerr << t.symbol() << " ";});
		//cerr << "\n";

		if (!tokenq.empty()) {
			stringstream ss;
			ss << "trailing tokens:";
			// Display up to 5 trailing tokens
			auto i = tokenq.begin();
			for (int k = 0; k < 5 && i != tokenq.end(); ++k, ++i) 
				ss << " " << i->symbol();
			throw parse_error( ss.str() );
		}


		//cerr << "RPN: ";
		//
		//for_each(rpn.cbegin(), rpn.cend(), [](const Token t) {
		//cerr << t.symbol() << " ";});
		//cerr << "\n";
		unique_ptr<Functor> root( postfix_to_tree(rpn) );
		if (!rpn.empty()) {
			string errmsg = "trailing tokens in RPN engine:";
			while (!rpn.empty()) {
				if (!rpn.front().is_null())
					errmsg += " " + rpn.front().symbol();
				else
					errmsg += " nullptr op";
				//delete rpn.front();
				rpn.pop_front();
			}
			throw parse_error(errmsg);
		}
		if (!root) throw parse_error("empty expression read");
		return root.release();
	}


	Token read_atom(istream& is, long long& linec, std::set<id_type>& qvars, const Dictionary& dic)
	{
		char ch;
		//cerr << "read_atom, reading in character\n";
		is.get(ch);
		//cerr << "read_atom, read: " << ch << "\n";
		if (ch == '\'') {
			// Note: this always defines an operator, i.e. prefix syntax
			// Note: includes the quotes '' when it would otherwise be a variable
			string name;
			for (;;) {
				is.get(ch);
				if (!is || is.eof()) {
					throw parse_error("at line " + to_string(linec) + ", unfinished string literal, expected \'");
				}
				if (ch == '\'') break;
				name += ch;
			}
			if (name.empty() || functor_map::is_variable(name)) {
				name.insert(name.begin(),'\'');
				name.push_back('\'');
			}
			// Note: symbols wrapped in '' are always functions or constants
			//std::cerr << "read_atom, name: " << name << "\n";
			return Token(name,Token::Function);
		} else if (isalpha(ch) || ch == '_') {
			// depending on context: Functor, function/predicate symbol, or built-in token
			// read name
			string name;
			name += ch;
			for (;;) {
				if (!is.get(ch)) break;
				if (!isalnum(ch) && ch != '_') {
					is.putback(ch);
					break;
				}
				name += ch;
			}
			// look up word in dictionary
			try {
				// try predefined operator
				// NOTE: this may not add the correct operator if there are overloads
				//cerr << "looking up in dictionary...";
				auto tok = dic.get(name);
				//cerr << "got token: " << tok.symbol() << "\n";
				return tok;
				//cerr << "found built-in " << name << "\n";
			} catch (dic_not_found) {
				//cerr << "caught not_found\n";
				if (islower(name[0])) {
					// function symbol
					//cerr << "adding functor \"" << name << "\"\n";
					return Token(name,Token::Function);
				} else { // isupper(name[0]) || name[0] == '_'
					if (name == "_") {
						// make unique variable
						// cerr << "making unique variable\n";
						return Token(Functor::unique_index(), Token::Variable);
					} else {
						//cerr << "adding variable \"" << name << "\"\n";
						// Index this variable
						//cerr << "variable is: " << vid << " -> " << fmap.get_data(vid) << "\n";
						qvars.insert(Functor::index(name));
						return Token(name, Token::Variable);
					}
				}
			}
			//cerr << "read word: " << name << "\n";
		} else if (isdigit(ch) || ch == '-' && isdigit(is.peek())) {
			string num(1, ch);
			bool dec = false;
			for (;;) {
				is.get(ch);
				if (!is) break;
				if (ch == '.' && dec == false && isdigit(is.peek())) {
					num += '.';
					dec = true;
					continue;
				}
				if (!isdigit(ch)) {
					is.unget();
					break;
				}
				num += ch; // ch is digit
			}
			// Convert to int/double
			//DEBUG_TRACE(cerr << "read numeric: " << num << ", has comma: " << dec << "\n");
			if (dec) {
				return Token::make_double(destringify<double>(num));
			} else {
				return Token::make_int(destringify<long long>(num));
			}
		} else {
			// predefined operators such as :-  <-  . etc
			// Read as many non-alphanumeric characters as possible
			static const string spec = ",()[|]";
			static const string forbid = "?%\'\"";
			//cerr << "Trying to read operator, ch: " << ch << "\n";
			string name(1,ch);
			if (std::find(spec.begin(),spec.end(),ch) != spec.end()) {
				// Special character: don't expand
				//cerr << "read atom special character: " << name << "\n";
				return Token(name,Token::Constant);
			} else if (std::find(forbid.begin(),forbid.end(),ch) != forbid.end()) {
				// Forbidden character, fail
				throw parse_error("at line " + to_string(linec) + ", invalid atom character " + ch);
			}
			//is.putback(ch);
			//cerr << "looking for predefined operator: " << ch << "\n";
			if (!is.eof()) {
				//cerr << "entering ch reading loop\n";
				for (;;) {
					is.get(ch);
					if (is.eof()) break;
					if (!is) {
						//cerr << "bad stream, eof: " << is.eof() << "\n";
						throw parse_error("at line " + to_string(linec) + ", bad stream while reading built-in");
					}
					//cerr << "   ch: " << ch << "\n";
					if (isalnum(ch) || isspace(ch) || std::find(spec.begin(),spec.end(),ch) != spec.end() 
						|| find(forbid.begin(),forbid.end(),ch) != forbid.end()) {
							is.putback(ch);
							break;
					} else {
						name += ch;
					}
				}
			}
			if (name.empty()) {
				throw parse_error("at line " + to_string(linec) + ", failed to read atom");
			}
			const Trie<Token>* tptr = dic.get_trie();
			//cerr << "Searching in trie: " << name << "\n";
			// Search for node in trie
			if (tptr && (tptr = (*tptr)[name]) != nullptr && tptr->is_entry()) {
				// Predefined operator
				//cerr << "read predefined operator: " << name << "\n";
				return *tptr->value_begin();
			} else {
				// Not a predefined operator, so make it a constant
				//cerr << "read special constant: " << name << "\n";
				return Token(name,Token::Constant);
			}

			////cerr << "predefined op? ch: \"" << ch << "\"\n";
			////DEBUG_TRACE(cerr << "reading predefined operator, ch: " << ch << "\n");
			//string name(1, ch);
			//const Trie<Token>* curr_node = dic.get_trie();
			//const Trie<Token>* next_node = curr_node->next(ch);
			//if (!next_node) {
			//	// Not a predefined operator, so make it a constant
			//	return Token(name,Token::Constant);
			//	//throw parse_error("at line " + to_string(linec) + ", unrecognized token: " + ch);
			//}
			//while (next_node) {
			//	is.get(ch);
			//	if (!is) throw parse_error("at line " + to_string(linec) + ", bad stream while reading built-in");
			//	if (is.eof()) break;
			//	//cerr << "read ch: " << ch << "\n";
			//	curr_node = next_node;
			//	next_node = next_node->next(ch);
			//	name += ch;
			//}
			//if (!is.eof()) {
			//	is.unget();
			//	// UPDATE: use name.pop_back() when gcc supports it (c++11 feature)
			//	name.erase(name.length()-1); // name.pop_back();
			//}
			//if (!curr_node->is_entry()) {
			//	// Not a predefined operator, so make it a constant
			//	return Token(name,Token::Constant);
			//	// throw parse_error("at line " + to_string(linec) + ", unrecognized token: " + name);
			//}
			//return **curr_node->value_begin();
			//// assert( tmp.symbol() == name );
			//// DEBUG_TRACE(cerr << "identified token: " << name << "\n");
			//// return tmp;

			//// Treat Progol operator ? as a termination operator (like '.')
			////if (name == "?") return tokenq;
		}
	}


	list<Token> tokenize(istream& is, long long& linec, std::set<id_type>& qvars, const Dictionary& dic) 
	{
		list<Token> tokenq;
		char ch = '\0';

		for (;;) {
			ch = is.get();
			// cerr << "tokenize, read: " << ch << "\n";
			if (is.eof()) {
				return tokenq; // end of input
			} else if (!is) {
				throw parse_error("bad input stream state");
			}
			// Store newlines as token for syntax error messages
			if (ch == '\n') { tokenq.push_back(Token(0,Token::Newline)); ++linec; continue; }
			// Skip whitespace
			if (isspace(ch)) continue;

			switch (ch) {
			case '.': {
				// end of input or list pair functor?
				ch = is.peek();
				if (is.eof() || ch != '(') return tokenq; // terminating dot
				else {
					// list pair functor
					tokenq.push_back(Token(pair_id, Token::Function));
				}
				break; // break case 
					  }
			case '%': {
				// Check if special comment
				ch = is.get();
				if (is.eof()) return tokenq;
				if (ch == '\n') { tokenq.push_back(Token(0,Token::Newline)); ++linec; break; } // end of empty comment
				if (ch == '!') {
					DEBUG_TRACE(cerr << "detected %!\n");
					ch = is.peek();
					if (ch == '%') {
						DEBUG_TRACE(cerr << "detected uncomment\n");
						// read %!%, which means end of line comment
						is.ignore(); // throw away %
						break; // continue reading this line
					} else if (ch == '{') {
						DEBUG_TRACE(cerr << "detected block comment\n");
						// read %!{, read until !}%
						is.ignore(); // throw away {
						// ignore until we read !}%
						for (;;) {
							ch = is.get();
							if (is.eof()) return tokenq;
							if (!is) throw parse_error("bad input stream state");
							if (ch == '!') {
								ch = is.peek();
								if (is.eof()) return tokenq;
								if (ch != '}') continue;
								is.ignore(); // ignore }
								ch = is.peek();
								if (is.eof()) return tokenq;
								if (ch != '%') continue;
								is.ignore(); // ignore %
								DEBUG_TRACE(cerr << "detected end of block comment\n");
								break;
							}
						}
						break; // done with comment block
					} // else: normal comment, fall through
				}
				// Comment, ignore rest of line
				while (!is.eof() && is.get() != '\n');
				if (is.eof()) return tokenq;
				tokenq.push_back(Token(0,Token::Newline)); // add new line
				++linec;
				break; // break case
					  }
			case '\"': { // convert string to list of numeric codes
				bool first = true;
				tokenq.push_back(Token(ll_id, Token::LeftList));
				for (;;) {
					is.get(ch);
					if (!is || is.eof()) {
						throw parse_error("at line " + to_string(linec) + ", unfinished string literal, expected \"");
					}
					if (ch == '\"') break;
					if (!first) tokenq.push_back(Token(sequence_id, Token::SeqSep, 1000));
					tokenq.push_back(Token(stringify(int(ch)), Token::Constant));
					first = false;
				}
				tokenq.push_back(Token(rl_id, Token::RightList));
				break; // break case
					   }
			case '(': {
				tokenq.push_back(Token(lp_id, Token::Leftp));
				break; // break case
					  }
			case ')': {
				tokenq.push_back(Token(rp_id, Token::Rightp));
				break; // break case
					  }
			case '[': {
				tokenq.push_back(Token(ll_id, Token::LeftList));
				break; // break case
					  }
			case ']': {
				tokenq.push_back(Token(rl_id, Token::RightList));
				break; // break case
					  }
			case '|': {
				tokenq.push_back(Token(lsep_id, Token::ListSep));
				break; // break case
					  }
			case '!': {
				// Cut elimination
				tokenq.push_back(Token(cut_id, Token::Constant));
				break; // break case
					  }
			case '$': 
				if (!dic.has("$")) {
					ch = is.peek();
					if (is.eof()) return tokenq;
					if (isdigit(ch)) {
						// Support conversion from integer form to s-form, e.g. $2 = s(s(0))
						unsigned n;
						is >> n; // read integer
						for (unsigned k = 0; k < n; ++k) {
							tokenq.push_back(Token(s_id, Token::Function));
							tokenq.push_back(Token(lp_id, Token::Leftp));
						}
						tokenq.push_back(Token(zero_id, Token::Constant));
						for (unsigned k = 0; k < n; ++k) {
							tokenq.push_back(Token(rp_id, Token::Rightp));
						}
						break; // break case
					} else goto default_case;
				}
			case ',':
				{
					tokenq.push_back(Token(sequence_id,Token::xfy,1000));
					break;
				}
			case '?': 
				{
					// cerr << "tokenize: read ?\n";
					tokenq.push_back(Token(qmark_id,Token::xf,1500));
					return tokenq;  
				}
			default: {
default_case:
				is.putback(ch);
				Token tok = read_atom(is,linec,qvars,dic);
				tokenq.push_back(std::move(tok));
				// Treat Progol operator '?' as a termination operator (like '.')
				//if (tokenq.back().symbol() == "?") return tokenq;
				break; } // case default
			} // switch
		} // main loop

		return tokenq;
	}



	void tokens_to_operators(list<Token>& tokenq, long long& linec, const Dictionary& dic) 
	{
		// Convert tokens to appropriate operators
		// -(1+2)   2-3   -(3,4)
		// (1*2)-(3+3)
		// 2-(-4)
		bool left_value = false; // have we read an expression to the left?
		/*
		nullary functions -> constants
		operators with function notation -> functions
		switch to correct type of operator in case there are overloads
		operators -> constants (if no other option exists)
		*/

		for (auto i = tokenq.begin(); i != tokenq.end(); ++i) {
			auto j = i;
			++j;
			if (i->is_newline()) {
				// Skip
				++linec;
			} else if (i->is_function()) {
				if (j == tokenq.end() || !j->is_lp()) {
					// this function is nullary => constant
					// cerr << "Converting from function to constant: " << i->symbol() << "\n";
					i->set_type(Token::Constant);
					left_value = true;
				} else {
					left_value = false; // this is a function
				}
			} else if (i->is_operator()) {
				if (i->is_seq_sep()) {
					// cannot convert , operator (but can be given as function using ',')
					left_value = false;
					continue;
				}
				//cerr << "Found operator " << i->symbol() << ", is it a binary op as function?\n";
				// First of all, check if operator is in function syntax
				// Note: if we have a left value, this is impossible
				if (!left_value && is_binop_as_function(i, tokenq.end(), dic)) {
					// cerr << i->symbol() << " is a binary op as function\n";
					i->set_type(Token::Function);
				} else {
					//cerr << "operator but not function syntax\n";
					// Decide if we have prefix/suffix/infix operator
					if (left_value) {
						// operator is infix or suffix
						if (j == tokenq.end() || j->is_rdel()) {
							// 3 op => suffix
							try {
								i->set_type(dic.suffix_op(i->symbol()));
							} catch (dic_not_found) {
								throw parse_error("at line " + to_string(linec) + ", no suffix operator " + i->symbol());
							}
						} else if (!j->is_operator()) {
							// infix
							try {
								i->set_type(dic.infix_op(i->symbol()));
							} catch (dic_not_found) {
								throw parse_error("at line " + to_string(linec) + ", no infix operator " + i->symbol());
							}
						} else {
							// X i_op j_op
							// case 1:  X infix function
							// case 2:  X infix prefix
							// case 3:  X suffix infix
							// case 4:  X infix constant (operator -> constant conversion)
							try {
								i->set_type(dic.infix_op(i->symbol()));
								if (is_binop_as_function(j, tokenq.end(), dic)) {
									// case 1
									// do nothing
								} else {
									// case 2
									j->set_type(dic.prefix_op(j->symbol()));
								}
							} catch (dic_not_found) {
								try {
									// case 3
									i->set_type(dic.suffix_op(i->symbol()));
									j->set_type(dic.infix_op(j->symbol()));
								} catch (dic_not_found) {
									// case 4: convert j to constant
									// Note: ',' ')' and ']' cannot be converted to constant
									if (j->is_rdel()) {
										throw parse_error("at line " + to_string(linec) + ", no matching operators for " + i->symbol() + " and " + j->symbol());
									}
									// cerr << "Converting " << j->symbol() << " to constant\n";
									j->set_type(Token::Constant);
								}
							}
						}
					} else { // left_value == false
						// operator is prefix
						try {
							i->set_type(dic.prefix_op(i->symbol()));
							// This could be prefix, but if next thing is ',' or ')', it is not
							if (j == tokenq.end() || j->is_rdel()) throw dic_not_found();
						} catch (dic_not_found) {
							// Convert operator to constant
							// cerr << "Converting (prefix) to constant: " << i->symbol() << "\n";
							i->set_type(Token::Constant);
							left_value = true;
							continue; // avoid setting left_value = false below
							// throw parse_error("at line " + to_string(linec) + ", no prefix operator " + i->symbol());
						}
					}
				} // end if(binop_as_function) {} else {
				left_value = false;
			} else if (i->is_lp()) {
				// left_value is unchanged
			} else if (i->is_ll() || i->is_list_sep()) {
				left_value = false;
			} else {
				// variable or constant or (right) list
				left_value = true;
			} // what kind of token?
		} // for
	}



	bool is_binop_as_function(
		list<Token>::const_iterator i,
		list<Token>::const_iterator end,
		const Dictionary& dic) {
			//cerr << "inside is_binop_as_function\n";
			if (i == end) return false;
			if (i->is_seq_sep()) return false; // , can never be cast to function implicitly
			// Can we cast i to infix operator?
			try {
				dic.infix_op(i->symbol());
			} catch (dic_not_found) {
				//cerr << "cannot cast " << i->symbol() << " to infix, so not binop as func\n";
				return false;
			}
			//cerr << "pass 1: binary operator exists\n";
			if (++i == end || !i->is_lp()) return false; // must have '(' after function symbol
			//cerr << "pass 2: LP exists\n";
			// find ',' counting nestings of paranthesis (not including first)
			int pnesting = 0; // paranthesis nesting
			int lnesting = 0; // list nesting
			while (++i != end) {
				//cerr << "Binop_as_function, checking " << i->symbol() << "\n";
				if (i->is_ll()) ++lnesting;
				else if (i->is_lp()) ++pnesting;
				else if (i->is_rp()) {
					if (pnesting == 0 && lnesting == 0) return false;
					--pnesting;
				} else if (i->is_seq_sep()) {
					//cerr << "Binary operator is in function format\n";
					return true;
				} else if (i->is_rl()) --lnesting;
			}
			return false; // not binary operator in prefix format
	}




	list<Token> infix_to_postfix(list<Token>& tokenq, long long& linec, id_type end1, id_type end2, id_type end3) 
	{
		//cerr << "inside infix_to_postfix\n";
		list<Token> output;
		stack<Token> op_stack;
		bool last_leaf = false; // error detection: was last token a constant/var/function?
		Token prev; // previous token

		while (!tokenq.empty()) {
			Token& cref = tokenq.front();
			if (!cref.is_newline() && (cref.id() == end1 || cref.id() == end2 || cref.id() == end3)) break;

			Token curr = tokenq.front(); // make a copy
			tokenq.pop_front(); // pop tokenq front
			//cerr << "curr token: " << curr.symbol() << "\n";

			if (curr.is_newline()) {
				++linec; // increase line count
			} else if (curr.is_variable()) {
				if (last_leaf) {
					throw parse_error("at line " + to_string(linec) + ", could not interpret " + prev.symbol() + " followed by variable " + curr.symbol());
				}
				last_leaf = true;
				//cerr << "pushing variable " << curr.symbol() << " to output\n";
				output.push_back(curr);
			} else if (curr.is_constant()) {
				if (last_leaf) {
					throw parse_error("at line " + to_string(linec) + ", could not interpret " + prev.symbol() + " followed by constant " + curr.symbol());
				}
				last_leaf = true;
				//cerr << "pushing constant " << curr.symbol() << " to output\n";
				output.push_back(curr);
			} else if (curr.is_function()) {
				if (last_leaf) {
					throw parse_error("at line " + to_string(linec) + ", could not interpret " + prev.symbol() + " followed by function " + curr.symbol());
				}
				last_leaf = true;
				//cerr << "recursively reading function arguments for " << curr.symbol() << "\n";
				// the argument list (a,b,c) should NOT be treated as a sequence
				if (tokenq.empty() || !tokenq.front().is_lp()) {
					throw parse_error("at line " + to_string(linec) + ", function " + curr.symbol() + " must have (");
				}
				tokenq.pop_front(); // ignore (
				output.push_back(Token()); // mark end of function arguments with nullptr
				for (;;) {
					list<Token> arg = infix_to_postfix(tokenq, linec, sequence_id, rp_id); // read arg
					if (arg.empty()) {
						throw parse_error("at line " + to_string(linec) + ", unexpected token " + tokenq.front().symbol() + " while reading function arguments");
					}
					//cerr << "read argument: ";
					//for_each(arg.begin(),arg.end(),[](Token t){cerr<<"\""<<t.symbol()<<"\"\n";});
					//cerr << "\n";
					output.splice(output.end(), arg); // add arg to output
					if (tokenq.empty()) {
						throw parse_error("at line " + to_string(linec) + ", function " + curr.symbol() + " is missing enclosing )");
					}
					if (tokenq.front().is_rp()) break;
					if (!tokenq.front().is_seq_sep()) {
						throw parse_error("at line " + to_string(linec) + ", function " + curr.symbol() + " has unexpected " + tokenq.front().symbol());
					}
					tokenq.pop_front(); // ignore ","
				}
				tokenq.pop_front(); // ignore ")"
				output.push_back(curr);
				//op_stack.top().set_type(Token::Function); // for operators used with function format
			} else if (curr.is_prefix_op()) {
				last_leaf = false;
				//cerr << "prefix op: " << curr.symbol() << "\n";
				if (!op_stack.empty()) {
					const Token& left = op_stack.top();
					if (left.is_fx() && left.precedence() <= curr.precedence()) {
						throw parse_error("at line " + to_string(linec) + ", " + left.symbol() + " is declared fx but has lower precedence than " + curr.symbol());
					}
				}
				op_stack.push(curr);
			} else if (curr.is_suffix_op()) {
				last_leaf = false;
				//cerr << "suffix op: " << curr.symbol() << "\n";
				if (curr.is_xf() && !output.empty()) {
					Token& left = output.back();
					if (left.is_suffix_op() && left.precedence() >= curr.precedence())
						throw parse_error("at line " + to_string(linec) + ", " + left.symbol() + " is declared xf but has higher precedence than " + curr.symbol());
				}
				// give priority to stack operators with lower precedence
				while (!op_stack.empty() && op_stack.top().is_operator()
					&& op_stack.top().precedence() < curr.precedence()) {
						output.push_back(op_stack.top());
						op_stack.pop();
				}
				output.push_back(curr); // move suffix op to output
			} else if (curr.is_binary_op()) {
				// Next token cannot be , or ) or ]
				last_leaf = false;
				//cerr << "binary op: " << curr.symbol() << "\n";
				while (!op_stack.empty()) {
					Token& left = op_stack.top();
					//cerr << "token left: " << left.symbol() << "\n";
					//if (left.is_seq_sep() || left.is_rp() || left.is_rl()) {
					//	throw parse_error("at line " + to_string(linec) + ", unexpected right operand " + left.symbol() + "\n");
					//}
					const int lp = left.precedence();
					const int cp = curr.precedence();
					if (left.is_binary_op() || left.is_prefix_op()) {
						if (curr.is_xfx() && left == curr) { // no need for lp >= cp
							throw parse_error("at line " + to_string(linec) + ", " + curr.symbol() + " is ambiguous with non-associative " + curr.symbol());
						} else if (curr.is_yfx() && lp <= cp || (curr.is_xfy() || curr.is_xfx()) && lp < cp) {
							// here we must swap the order of the operations
							//cerr << "outputting " << left.symbol() << "\n";
							output.push_back(left);
							op_stack.pop();
							continue;
						}
					}
					break; // nothing more to push back
				}
				//cerr << "pushing infix operator " << curr.symbol() << " to stack\n";
				op_stack.push(curr);
			} else if (curr.is_ll()) {
				if (last_leaf) {
					throw parse_error("at line " + to_string(linec) + ", " + prev.symbol() + " right before [");
				}
				last_leaf = true;
				// [a,b,c] => p(a,p(b,p(c,[]))) => nullptr a nullptr b nullptr c [] . . .
				//cerr << "recursively reading list arguments\n";
				if (tokenq.empty()) throw parse_error("at line " + to_string(linec) + ", unexpected end of tokens while reading list");
				if (tokenq.front().is_rl()) {
					tokenq.pop_front(); // ignore ]
					output.push_back(Token(empty_list_id, Token::Constant));
				} else {
					unsigned lsize = 0;
					for (;;) {
						list<Token> arg = infix_to_postfix(tokenq, linec, sequence_id, lsep_id, rl_id); // read arg
						++lsize;
						//cerr << "read argument: ";
						//for_each(arg.begin(),arg.end(),[](const Token& t){cerr<<t.symbol()<<" ";});
						//cerr << "\n";
						if (tokenq.empty()) throw parse_error("at line " + to_string(linec) + ", no more tokens while reading list");
						id_type send = tokenq.front().id();
						if (send == sequence_id || send == rl_id || send == lsep_id) {
							tokenq.pop_front(); // ignore send token
							output.push_back(Token()); // nullptr before every constant
							output.splice(output.end(), arg); // add arg to output
							if (send == rl_id) {
								output.push_back(Token(empty_list_id, Token::Constant));
								break;
							} else if (send == lsep_id) {
								if (tokenq.empty()) {
									throw parse_error("at line " + to_string(linec) + ", expected list tail after |");
								}
								list<Token> tail = infix_to_postfix(tokenq, linec, rl_id);
								output.splice(output.end(), tail);
								tokenq.pop_front(); // pop "]"
								break;
							} // else send == ',' so continue
						} else {
							throw parse_error("at line " + to_string(linec) + ", expected |, ] or , while reading list");
						}
					}
					// add '.'
					while (lsize-- > 0) {
						output.push_back(Token(pair_id, Token::Function));
					}
				}
			} else if (curr.is_list_sep()) {
				last_leaf = false;
				// flush stack to output until [
				while (!op_stack.empty()) {
					if (op_stack.top().is_ll()) break;
					output.push_back(op_stack.top());
					op_stack.pop();
				}
				if (op_stack.empty()) throw parse_error("at line " + to_string(linec) + ", | can only be used inside list");
				output.push_back(curr); // push | to output
			} else if (curr.is_lp()) {
				last_leaf = false;
				//cerr << "pushing LP to stack\n";
				//op_stack.push(curr);
				// recursive call
				list<Token> subout = infix_to_postfix(tokenq, linec, rp_id);
				output.splice(output.end(), subout); // add subout to end of output
				if (tokenq.empty() || !tokenq.front().is_rp()) throw parse_error("at line " + to_string(linec) + ", missing )");
				tokenq.pop_front(); // throw away ")"
			} else {
				throw parse_error("at line " + to_string(linec) + ", unrecognized token: " + curr.symbol());
			}

			prev = move(curr); // set current token to previous
		} // while tokenq is not empty

		// move whatever is left
		while (!op_stack.empty()) {
			if (op_stack.top().is_lp() || op_stack.top().is_rp())
				throw parse_error("at line " + to_string(linec) + ", paranthesis mismatch");
			//cerr << "moving " << op_stack.top().symbol() << " to stack\n";
			output.push_back(op_stack.top());
			op_stack.pop();
		}

		return output;
	}



	Functor* postfix_to_tree(list<Token>& tokenq) 
	{
		if (tokenq.empty()) throw parse_error("postfix_to_tree: missing tokens");
		Token curr = tokenq.back();

		if (curr.is_null()) throw parse_error("postfix_to_tree: nullptr read");

		unique_ptr<Functor> subroot;

		// Handle list notations
		if (curr.is_rl()) {
			//cerr << "found ], beginning of list\n";
			// ] tells us that we need to convert a sequence into a list until we find [
			tokenq.pop_back(); // pop ]
			if (tokenq.empty()) throw parse_error("could not match ] with [");
			//cerr << "next token is: " << tokenq.back().symbol() << "\n";
			if (tokenq.back().is_ll()) {
				tokenq.pop_back(); // pop [
				return new Functor(empty_list_id); // empty list
			} else {
				subroot.reset( new Functor(pair_id) );
				// check if we have | (we don't require tail to be variable)
				auto j = tokenq.rbegin();
				if (++j != tokenq.rend() && j->is_list_sep()) {
					// we have [...|V]
					subroot->steal(new Functor(tokenq.back().id())); // add variable (or whatever)
					tokenq.pop_back(); // pop variable (or whatever)
					tokenq.pop_back(); // pop |
				} else {
					subroot->steal(new Functor(empty_list_id)); // terminate list with []
				}
				int args = 1; // args = 1+(number of commas)
				for (Token* ptr = &tokenq.back(); ptr->is_seq_sep(); ptr = &tokenq.back()) {
					++args;
					tokenq.pop_back();
					if (tokenq.empty()) throw parse_error("premature end while reading arguments");
				}
				//cerr << "list has : " << args << " elements\n";
				// build list bottom-up
				subroot->steal(postfix_to_tree(tokenq), 0); // read first argument
				for (int i = 1; i < args; ++i) {
					unique_ptr<Functor> n( new Functor(pair_id) );
					n->steal(postfix_to_tree(tokenq));
					n->steal(subroot.release());
					subroot = move(n); // let subtree point to highest node so far
				}
				if (tokenq.empty()) throw parse_error("expected [ but no more tokens");
				if (!tokenq.back().is_ll()) throw parse_error("expected [ but read " + tokenq.back().symbol());
				tokenq.pop_back(); // pop [
				return subroot.release();
			}
		}

		// Handle sequences: store (a,b,c) as (, a b c)
		if (curr.is_binary_op() && curr.is_seq_sep()) {
			// ) tells us that we need to convert a sequence (a,b,c) into ','(a,','(b,c))
			tokenq.pop_back(); // pop )
			if (tokenq.empty()) throw parse_error("could not match ) with (");
			//cerr << "next token is: " << tokenq.back().symbol() << "\n";
			unsigned seq_length = 2;
			while (!tokenq.empty() && tokenq.back().is_binary_op() && tokenq.back().is_seq_sep()) {
				++seq_length;
				tokenq.pop_back();
			}
			//cerr << "seq_length: " << seq_length << "\n";
			//subroot.reset( new Functor(sequence_id) );
			subroot.reset(new Functor(sequence_id));
			subroot->steal(postfix_to_tree(tokenq)); // last element (n:th)
			subroot->steal_front(postfix_to_tree(tokenq)); // n-1:th element
			seq_length -= 2;
			while (seq_length-- > 0) {
				unique_ptr<Functor> tmp( new Functor(sequence_id) );
				tmp->steal(postfix_to_tree(tokenq));
				tmp->steal(subroot.release());
				subroot = move(tmp);
			}
			return subroot.release();
		}

		// Handle other cases (not list or sequence)

		// Create subroot node
		subroot.reset( new Functor(curr.id()) );

		if (curr.is_constant() || curr.is_variable()) {
			//cerr << "read constant/variable " << curr.symbol() << "\n";
			tokenq.pop_back();
		} else if (curr.is_function()) {
			//cerr << "read functor " << curr.symbol() << "\n";
			tokenq.pop_back();
			// read until nullptr
			while (!tokenq.empty()) {
				if (tokenq.back().is_null()) break;
				subroot->steal_front(postfix_to_tree(tokenq));
			}
			if (tokenq.empty()) throw parse_error("expected function arguments");
			tokenq.pop_back(); // ignore nullptr
		} else if (curr.is_prefix_op() || curr.is_suffix_op()) {
			//cerr << "read prefix/suffix " << curr.symbol() << "\n";
			// read one argument
			tokenq.pop_back();
			unique_ptr<Functor> child( postfix_to_tree(tokenq) );
			if (child == nullptr) throw parse_error("read nullptr");
			subroot->steal(child.release());
		} else if (curr.is_binary_op()) {
			//cerr << "read infix operator " << curr.symbol() << "\n";
			tokenq.pop_back(); // pop binary op
			// read two arguments (they come in reverse order due to RPN)
			subroot->steal(postfix_to_tree(tokenq));
			subroot->steal_front(postfix_to_tree(tokenq));
			//cerr << "subroot: " << *subroot << "\n";
		} else {
			// it's not a function
			throw parse_error("unrecognized symbol");
		}

		return subroot.release();
	}


	Dictionary::Dictionary(Prolog_syntax) : tokt(new Trie<Token>) 
	{
		add(Token(lp_id, Token::Leftp));
		add(Token(rp_id, Token::Rightp));

		// For Progol compatibility: use suffix ? for queries
		add(Token(qmark_id, Token::xf, 1500)); // Progol style query

		add(Token(if_id, Token::xfx, 1200)); // binary if
		add(Token(if_id, Token::fx, 1200)); // unary if
		add(Token(query_id, Token::fx, 1200)); // query

		add(Token(implies_id, Token::xfx, 1150)); // '=>'/2
		add(Token(prevents_id, Token::xfx, 1150)); // prevents/2
		add(Token(prevent_id, Token::fx, 1150)); // prevents/1

		add(Token(or_id, Token::xfy, 1100)); // or
		add(Token(ifthen_id, Token::xfy, 1050)); // if-then
		add(Token(sequence_id, Token::xfy, 1000)); // and

		add(Token(naf_id, Token::fy, 900)); // negation as failure
		add(Token(not_id, Token::fy, 900)); // negation as failure

		add(Token(plus_id, Token::yfx, 500));
		add(Token(mul_id, Token::yfx, 400));
		add(Token(minus_id, Token::yfx, 500));
		add(Token(div_id, Token::yfx, 400));
		add(Token(pow_id, Token::yfx, 200));
		add(Token(star2_id, Token::yfx, 200));
		add(Token(minus_id, Token::fx, 500)); // for mode declarations and unary minus
		add(Token(plus_id, Token::fx, 500)); // for mode declarations (input)
		add(Token(hash_id, Token::fx, 500)); // for mode declarations (ground)

		add(Token(univ_id, Token::xfx, 700)); // univ
		add(Token(syn_equal_id, Token::xfx, 700)); // syntactic equality
		add(Token(syn_unequal_id, Token::xfx, 700)); // syntactic inequality
		add(Token(is_id, Token::xfx, 700)); // is operator

		add(Token(math_equal_id, Token::xfx, 700)); // arithmetic equality
		add(Token(math_unequal_id, Token::xfx, 700)); // arithmetic unequality
		add(Token(lt_id, Token::xfx, 700)); // arithmetic less than
		add(Token(gt_id, Token::xfx, 700)); // arithmetic greater than
		add(Token(le_id, Token::xfx, 700)); // arithmetic less equal
		add(Token(ge_id, Token::xfx, 700)); // arithmetic greater equal

		add(Token(equal_id, Token::xfx, 700)); // unification
		add(Token(unequal_id, Token::xfx, 700)); // is_not_unifiable

		add(Token(lessat_id, Token::xfx, 700)); // before
		add(Token(lesseqat_id, Token::xfx, 700)); // before or same
		add(Token(greaterat_id, Token::xfx, 700)); // after equal
		add(Token(greatereqat_id, Token::xfx, 700)); // after or same
		add(Token(isvariant_id, Token::xfx, 700)); // after or same
		add(Token(isnotvariant_id, Token::xfx, 700)); // after or same
	}


}

